# ============================================================
# Home Data for ML Course (Kaggle Learn Users)
# CatBoost版：12000未満を狙う（入門コンペで相性が良い）
#
# 重要ポイント
#  - CatBoostは「カテゴリ列をそのまま」扱える（One-Hot不要）
#  - 欠損処理やカテゴリ処理が比較的ラクで強い
#  - KFoldで複数モデルを学習 → test予測を平均して安定化
#  - 追加で log1p(SalePrice) で学習して、最後に expm1 で戻す
#    → 高額物件の影響を弱めてRMSEが下がりやすい（効くことが多い）
#
# 実行方法
#  - Kaggle Notebook の1セルにコピペして実行
#  - 実行後に submission.csv ができます
#
# ※初回だけ catboost のインストールが必要な場合があります：
#   !pip -q install catboost
# ============================================================

import numpy as np
import pandas as pd

from sklearn.model_selection import KFold
from sklearn.metrics import mean_squared_error

# --- CatBoost ---
from catboost import CatBoostRegressor, Pool


# ============================================================
# 0) データ読み込み
# ============================================================
train = pd.read_csv("/kaggle/input/home-data-for-ml-course/train.csv")
test  = pd.read_csv("/kaggle/input/home-data-for-ml-course/test.csv")

# ============================================================
# 1) 外れ値除去（効くことが多い定番）
#    「超広いのに安い」家はモデルを変に引っ張ることがある
# ============================================================
train = train.drop(train[(train["GrLivArea"] > 4000) & (train["SalePrice"] < 300000)].index)

y = train["SalePrice"].copy()
X = train.drop(columns=["SalePrice"]).copy()
X_test = test.copy()

# ============================================================
# 2) 特徴量追加（軽量だが効きやすい派生を少しだけ）
#    ※CatBoostはそのままでも強いが、派生は上積みになりやすい
# ============================================================
def add_features(df: pd.DataFrame) -> pd.DataFrame:
    df = df.copy()

    # 面積系は「ない」=0 が自然な列が多い（地下室なし等）
    for c in ["TotalBsmtSF", "1stFlrSF", "2ndFlrSF", "GarageArea"]:
        if c in df.columns:
            df[c] = df[c].fillna(0)

    # 合計面積（かなり強い）
    df["TotalSF"] = df.get("TotalBsmtSF", 0) + df.get("1stFlrSF", 0) + df.get("2ndFlrSF", 0)

    # 築年数・改築後年数（強い）
    if "YrSold" in df.columns and "YearBuilt" in df.columns:
        df["HouseAge"] = df["YrSold"] - df["YearBuilt"]
    if "YrSold" in df.columns and "YearRemodAdd" in df.columns:
        df["RemodAge"] = df["YrSold"] - df["YearRemodAdd"]

    # 風呂合計（地味に効く）
    for c in ["FullBath", "HalfBath", "BsmtFullBath", "BsmtHalfBath"]:
        if c in df.columns:
            df[c] = df[c].fillna(0)
    if all(c in df.columns for c in ["FullBath","HalfBath","BsmtFullBath","BsmtHalfBath"]):
        df["TotalBath"] = df["FullBath"] + 0.5*df["HalfBath"] + df["BsmtFullBath"] + 0.5*df["BsmtHalfBath"]

    # 品質×面積（強い）
    if "OverallQual" in df.columns and "GrLivArea" in df.columns:
        df["Qual_x_GrLivArea"] = df["OverallQual"] * df["GrLivArea"]
    if "OverallQual" in df.columns:
        df["Qual_x_TotalSF"] = df["OverallQual"] * df["TotalSF"]

    return df

X = add_features(X)
X_test = add_features(X_test)

# ============================================================
# 3) カテゴリ列を取得（CatBoostに渡すため）
#    CatBoostは「カテゴリ列のインデックス」を指定できる
# ============================================================
cat_cols = X.select_dtypes(include=["object"]).columns.tolist()
cat_idx = [X.columns.get_loc(c) for c in cat_cols]

print("Categorical columns:", len(cat_cols))

# ============================================================
# 4) 欠損処理（最低限）
# ------------------------------------------------------------
# CatBoostは欠損に比較的強いが、
# - objectのNaNは明示的に "Missing" などに置換する方が安全
# - 数値はそのままでも動くが、極端に欠損が多い列があるので
#   “median補完”を入れると安定することが多い
# ============================================================
# カテゴリ欠損は文字に置換
X[cat_cols] = X[cat_cols].fillna("Missing")
X_test[cat_cols] = X_test[cat_cols].fillna("Missing")

# 数値欠損は中央値補完（trainの中央値でtestも埋める）
num_cols = X.columns.difference(cat_cols).tolist()
med = X[num_cols].median()
X[num_cols] = X[num_cols].fillna(med)
X_test[num_cols] = X_test[num_cols].fillna(med)

# ============================================================
# 5) 目的変数を log1p で学習（効くことが多い）
# ------------------------------------------------------------
# 学習：log価格
# 予測：expm1で元のドルに戻す
#
# ※この入門コンペはLB表示がドルRMSEですが、
#   log学習→expm1提出がRMSEを下げることが割とあります。
# ============================================================
y_log = np.log1p(y)

# ============================================================
# 6) KFold 学習 + 予測平均
# ============================================================
kf = KFold(n_splits=5, shuffle=True, random_state=42)

oof_log = np.zeros(len(X))
test_pred = np.zeros(len(X_test))

for fold, (tr_idx, va_idx) in enumerate(kf.split(X), start=1):
    X_tr, X_va = X.iloc[tr_idx], X.iloc[va_idx]
    y_tr, y_va = y_log.iloc[tr_idx], y_log.iloc[va_idx]

    # CatBoost用のPool（カテゴリ列を指定）
    train_pool = Pool(X_tr, y_tr, cat_features=cat_idx)
    valid_pool = Pool(X_va, y_va, cat_features=cat_idx)
    test_pool  = Pool(X_test, cat_features=cat_idx)

    # --------------------------------------------------------
    # CatBoostの設定（強めだが過学習しにくい寄り）
    # - loss_function: RMSE（回帰）
    # - depth: 木の深さ（6〜10あたりが定番）
    # - learning_rate小さめ + iterations多め
    # - early_stoppingで自動停止（過学習抑制＆高速化）
    # --------------------------------------------------------
    model = CatBoostRegressor(
        loss_function="RMSE",
        iterations=20000,          # 早期終了するので多めに置いてOK
        learning_rate=0.03,
        depth=8,
        l2_leaf_reg=3.0,
        random_seed=42,
        eval_metric="RMSE",
        od_type="Iter",
        od_wait=300,               # 300回改善しなければ停止
        verbose=300                # 進捗表示（うるさければ 0 に）
    )

    model.fit(train_pool, eval_set=valid_pool, use_best_model=True)

    # 検証予測（log空間）
    pred_va_log = model.predict(valid_pool)
    oof_log[va_idx] = pred_va_log

    # foldのRMSE（ドル空間）を表示したいのでexpで戻す
    pred_va = np.expm1(pred_va_log)
    y_va_price = np.expm1(y_va)
    fold_rmse = mean_squared_error(y_va_price, pred_va, squared=False)
    print(f"[Fold {fold}] RMSE(price): {fold_rmse:.2f}")

    # test予測（log→priceへ戻して平均）
    pred_test_log = model.predict(test_pool)
    test_pred += np.expm1(pred_test_log) / kf.n_splits

# 全体CV（ドル空間）
cv_rmse = mean_squared_error(np.expm1(y_log), np.expm1(oof_log), squared=False)
print(f"\n[CV] RMSE(price): {cv_rmse:.2f}")

# ============================================================
# 7) submission 作成
# ============================================================
submission = pd.DataFrame({
    "Id": test["Id"],
    "SalePrice": test_pred
})
submission.to_csv("submission.csv", index=False)
print("✅ saved: submission.csv")
